# == Class: iop::service_vulnerability
#
# Install and configure the Vulnerability Engine services
#
# === Parameters:
#
# $image::                     The container image
#
# $ensure::                    Ensure service is present or absent
#
# $database_user:: Username for the vulnerability database
#
# $database_name:: Name of the vulnerability database
#
# $database_password:: Password for the vulnerability database
#
# $taskomatic_jobs:: Jobs of taskomatic in format "jobname:minutes_interval" separated by comma.
#
# $taskomatic_startup:: Jobs to be scheduled on startup separated by comma.
#
class iop::service_vulnerability (
  String[1] $image                    = 'quay.io/iop/vulnerability-engine:latest',
  Enum['present', 'absent'] $ensure   = 'present',
  String[1] $database_name            = 'vulnerability_db',
  String[1] $database_user            = 'vulnerability_admin',
  String[1] $database_password        = extlib::cache_data('iop_cache_data', 'vulnerability_db_password', extlib::random_password(32)),
  String[1] $taskomatic_jobs          = 'stale_systems:5,delete_systems:30,cacheman:5',
  String[1] $taskomatic_startup       = 'cacheman',
) {
  include podman
  include iop::database
  include iop::core_kafka
  include iop::core_network
  include iop::core_host_inventory
  include iop::service_vmaas

  $service_name = 'iop-service-vulnerability'
  $database_username_secret_name = "${service_name}-database-username"
  $database_password_secret_name = "${service_name}-database-password"
  $database_name_secret_name = "${service_name}-database-name"
  $database_host_secret_name = "${service_name}-database-host"
  $database_port_secret_name = "${service_name}-database-port"

  podman::secret { $database_username_secret_name:
    ensure => $ensure,
    secret => Sensitive($database_user),
  }

  podman::secret { $database_password_secret_name:
    ensure => $ensure,
    secret => Sensitive($database_password),
  }

  podman::secret { $database_name_secret_name:
    ensure => $ensure,
    secret => Sensitive($database_name),
  }

  podman::secret { $database_host_secret_name:
    ensure => $ensure,
    secret => Sensitive('/var/run/postgresql'),
  }

  podman::secret { $database_port_secret_name:
    ensure => $ensure,
    secret => Sensitive('5432'),
  }

  # Prevents errors if run from /root etc.
  Postgresql_psql {
    cwd => '/',
  }

  include postgresql::client, postgresql::server

  postgresql::server::db { $database_name:
    user     => $database_user,
    password => postgresql::postgresql_password($database_user, $database_password),
    owner    => $database_user,
    encoding => 'utf8',
    locale   => 'en_US.utf8',
  }

  iop::postgresql_fdw { 'vulnerability':
    database_name        => $database_name,
    database_user        => $database_user,
    database_password    => $database_password,
    remote_database_name => $iop::core_host_inventory::database_name,
    remote_user          => $iop::core_host_inventory::database_user,
    remote_password      => $iop::core_host_inventory::database_password,
    require              => [
      Postgresql::Server::Db[$database_name],
      Postgresql::Server::Schema['inventory'],
      Postgresql_psql['create_or_replace_remote_view_inventory_hosts'],
    ],
  }

  podman::quadlet { 'iop-service-vuln-dbupgrade':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Iop::Postgresql_fdw['vulnerability'],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Database Upgrade Init Container',
      },
      'Service'   => {
        'Type'            => 'oneshot',
        'RemainAfterExit' => 'true',
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-dbupgrade',
        'Network'       => 'iop-core-network',
        'Exec'          => 'bash -c /engine/dbupgrade.sh',
        'Volume'        => [
          '/var/run/postgresql:/var/run/postgresql:rw',
        ],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'DISABLE_RBAC=TRUE',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
    },
  }

  podman::quadlet { 'iop-service-vuln-manager':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Iop::Postgresql_fdw['vulnerability'],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Manager Service',
        'After'       => ['network-online.target', 'iop-service-vuln-dbupgrade.service'],
        'Requires'    => 'iop-service-vuln-dbupgrade.service',
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-manager',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh manager',
        'Volume'        => [
          '/var/run/postgresql:/var/run/postgresql:rw',
        ],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'DISABLE_RBAC=TRUE',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-taskomatic':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Taskomatic Service',
        'Wants'       => ['iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-taskomatic',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh taskomatic',
        'Volume'        => [
          '/var/run/postgresql:/var/run/postgresql:rw',
        ],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'IS_FEDRAMP=true',
          "JOBS=${taskomatic_jobs}",
          "JOBS_STARTUP=${taskomatic_startup}",
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-grouper':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Grouper Service',
        'Wants'       => ['iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-grouper',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh grouper',
        'Volume'        => ['/var/run/postgresql:/var/run/postgresql:rw'],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'KAFKA_HOST=iop-core-kafka',
          'KAFKA_PORT=9092',
          'KAFKA_GROUP_ID=vulnerability-grouper',
          'PAYLOAD_TRACKER_TOPIC=platform.payload-status',
          'GROUPER_INVENTORY_TOPIC=vulnerability.grouper.inventory.upload',
          'GROUPER_ADVISOR_TOPIC=vulnerability.grouper.advisor.upload',
          'PROMETHEUS_PORT=8085',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-listener':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Listener Service',
        'Wants'       => ['iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-listener',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh listener',
        'Volume'        => ['/var/run/postgresql:/var/run/postgresql:rw'],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'KAFKA_HOST=iop-core-kafka',
          'KAFKA_PORT=9092',
          'KAFKA_GROUP_ID=vulnerability-listener2',
          'EVENTS_TOPIC=platform.inventory.events',
          'PAYLOAD_TRACKER_TOPIC=platform.payload-status',
          'ADVISOR_RESULTS_TOPIC=platform.engine.results',
          'MESSAGE_TOPIC=vulnerability.evaluator.upload',
          'ALLOWED_REPORTERS=puptoo,satellite',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-evaluator-recalc':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Evaluator (Recalc) Service',
        'Wants'       => ['iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-evaluator-recalc',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh evaluator',
        'Volume'        => ['/var/run/postgresql:/var/run/postgresql:rw'],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'KAFKA_HOST=iop-core-kafka',
          'KAFKA_PORT=9092',
          'KAFKA_GROUP_ID=vulnerability',
          'PAYLOAD_TRACKER_TOPIC=platform.payload-status',
          'REMEDIATION_UPDATES_TOPIC=platform.remediation-updates.vulnerability',
          'EVALUATOR_RESULTS_TOPIC=vulnerability.evaluator.results',
          'EVALUATOR_TOPIC=vulnerability.evaluator.recalc',
          'VMAAS_HOST=http://iop-service-vmaas-webapp-go:8000',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-evaluator-upload':
    ensure       => $ensure,
    quadlet_type => 'container',
    user         => 'root',
    require      => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings     => {
      'Unit'      => {
        'Description' => 'Vulnerability Evaluator (Upload) Service',
        'Wants'       => ['iop-service-vuln-grouper.service', 'iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vuln-grouper.service', 'iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-evaluator-upload',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh evaluator',
        'Volume'        => ['/var/run/postgresql:/var/run/postgresql:rw'],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'KAFKA_HOST=iop-core-kafka',
          'KAFKA_PORT=9092',
          'KAFKA_GROUP_ID=vulnerability',
          'PAYLOAD_TRACKER_TOPIC=platform.payload-status',
          'REMEDIATION_UPDATES_TOPIC=platform.remediation-updates.vulnerability',
          'EVALUATOR_RESULTS_TOPIC=vulnerability.evaluator.results',
          'EVALUATOR_TOPIC=vulnerability.evaluator.upload',
          'VMAAS_HOST=http://iop-service-vmaas-webapp-go:8000',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Install'   => { 'WantedBy' => 'default.target' },
    },
  }

  podman::quadlet { 'iop-service-vuln-vmaas-sync':
    ensure         => $ensure,
    quadlet_type   => 'container',
    user           => 'root',
    service_ensure => 'stopped',
    require        => [
      Podman::Network['iop-core-network'],
      Postgresql::Server::Db[$database_name],
      Podman::Secret[$database_username_secret_name],
      Podman::Secret[$database_password_secret_name],
      Podman::Secret[$database_name_secret_name],
      Podman::Secret[$database_host_secret_name],
      Podman::Secret[$database_port_secret_name],
    ],
    settings       => {
      'Unit'      => {
        'Description' => 'Vulnerability VMAAS Sync Job',
        'Wants'       => ['iop-service-vmaas-webapp-go.service', 'iop-service-vuln-manager.service'],
        'After'       => ['iop-service-vmaas-webapp-go.service', 'iop-service-vuln-manager.service'],
      },
      'Container' => {
        'Image'         => $image,
        'ContainerName' => 'iop-service-vuln-vmaas-sync',
        'Network'       => 'iop-core-network',
        'Exec'          => '/engine/entrypoint.sh vmaas-sync',
        'Volume'        => ['/var/run/postgresql:/var/run/postgresql:rw'],
        'Environment'   => [
          'UNLEASH_BOOTSTRAP_FILE=develfeatureflags.json',
          'KAFKA_HOST=iop-core-kafka',
          'KAFKA_PORT=9092',
          'KAFKA_GROUP_ID=vulnerability',
          'MESSAGE_TOPIC=vulnerability.evaluator.recalc',
          'VMAAS_HOST=http://iop-service-vmaas-webapp-go:8000',
        ],
        'Secret'        => [
          "${database_username_secret_name},type=env,target=POSTGRES_USER",
          "${database_password_secret_name},type=env,target=POSTGRES_PASSWORD",
          "${database_name_secret_name},type=env,target=POSTGRES_DB",
          "${database_host_secret_name},type=env,target=POSTGRES_HOST",
          "${database_port_secret_name},type=env,target=POSTGRES_PORT",
        ],
      },
      'Service'   => {
        'Type'            => 'oneshot',
      },
      'Install'   => {
        'WantedBy'        => [],
      },
    },
  }

  $timer_ensure = $ensure ? { 'present' => true, default => false }

  systemd::timer { 'iop-service-vuln-vmaas-sync.timer':
    ensure        => $ensure,
    enable        => $timer_ensure,
    active        => $timer_ensure,
    service_unit  => 'iop-service-vuln-vmaas-sync.service',
    timer_content => file('iop/iop-service-vuln-vmaas-sync.timer'),
    require       => Podman::Quadlet['iop-service-vuln-vmaas-sync'],
  }
}
